#!/usr/bin/python
""" 
    Copyright (C) <2021-2026>  <Scorch>              
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
import sys, math, time
from OpenGL import GL, GLU, GLUT #pip install pyopengl
from OpenGL.arrays.vbo import VBO
import OpenGL.GL.shaders
import ctypes

from pyopengltk import OpenGLFrame #pip install pyopengltk
from stl import mesh
import numpy
import math

class EZ_OpenGL_Frame(OpenGLFrame):
    def init_once(self):
        #print("init_once")
        self.tri_col  = (0,1,0,0)
        self.edge_col = (0,0,1,0)
        try:
            self.back_col
        except:
            self.back_col = (0,0,0,0)
        self.draw_faces  = True
        self.draw_edges  = False 
        self.draw_box    = False
        self.draw_axes   = False
        self.draw_paths  = True
        self.initialized = True
        self.ecoords=[]
        self.BOUNDS=(0,0,0,0,0,0)
        self.init_VBO()
        ############################################################
        ## Read shaders from files if they have not been read yet ##
        ############################################################

        self.fragment_shader_text="""
        uniform vec3 u_LightPos;     // The position of the light in eye space.
        uniform float u_Size;        // Size of the model for diffusion setting
          
        varying vec3 v_Position;     // Interpolated position for this fragment.
        varying vec4 v_Color;        // This is the color from the vertex shader interpolated across the triangle per fragment.
        varying vec3 v_Normal;       // Interpolated normal for this fragment.


        void main()                    		
        {                              
            // Will be used for attenuation.
            float distance = length(u_LightPos - v_Position);	
                
            // Get a lighting direction vector from the light to the vertex.
            vec3 lightVector = normalize(u_LightPos - v_Position);              	

            // Calculate the dot product of the light vector and vertex normal. If the normal and light vector are
            // pointing in the same direction then it will get max illumination.
            float diffuse;

            if (gl_FrontFacing)
            {
                diffuse = max(dot(v_Normal, lightVector), 0.0);
            }
            else 
            {
                diffuse = max(dot(-v_Normal, lightVector), 0.0);
            }               	  		  													  

            // Add attenuation. 
            diffuse = diffuse * (1.0 / (1.0 + (0.25 * distance/u_Size)));
            
            // Add ambient lighting
            diffuse = diffuse + 0.3;

            // Multiply the color by the diffuse illumination level to get final output color.
            gl_FragColor = (v_Color * diffuse);   
            
        }                                                                     	
        """
        self.vertex_shader_text="""
        uniform mat4 u_MVPMatrix;		// A constant representing the combined model/view/projection matrix.      		       
        uniform mat4 u_MVMatrix;		// A constant representing the combined model/view matrix.       		
        uniform vec4 a_Color;			// A constant representing information we will pass in. 	
        uniform int u_const_color;              // flag to use constant color vs. per pixel color
        uniform int u_const_normal;             // Flag to use normal toward eye vs per pixel normals

        varying vec3 v_Position;		// This will be passed into the fragment shader.       		
        varying vec4 v_Color;			// This will be passed into the fragment shader.          		
        varying vec3 v_Normal;			// This will be passed into the fragment shader.  	
                          
        // The entry point for our vertex shader.  
        void main()                                                 	
        {          
            // Transform the vertex into eye space. 	
            v_Position = vec3(u_MVMatrix * gl_Vertex);

            if (u_const_color==1) v_Color = a_Color;// Pass through the constant color.
            else                  v_Color = gl_Color;// Use per vertex color values
            
            // Transform the normal's orientation into eye space.
            if (u_const_normal==0) v_Normal = vec3(u_MVMatrix * vec4(gl_Normal, 0.0));
            else                   v_Normal = vec3(0,0,1);
                  
            // gl_Position is a special variable used to store the final position.
            // Multiply the vertex by the matrix to get the final point in normalized screen coordinates.
            gl_Position = u_MVPMatrix * gl_Vertex; 

        }                       
        """
        ############################################################
        self.DV=0.0
        self.DH=0.0
        
        GL.glEnable(GL.GL_DEPTH_TEST);       # enables depth testing
        GL.glDepthFunc(GL.GL_LEQUAL);        # the type of depth test to do
        self.Create_and_Compile_Shader()


    def set_tri_col(self,color=(0,1,0,0)):
        self.tri_col=(color[0]/255.0,color[1]/255.0,color[2]/255.0,color[3]/255.0)

    def set_edge_col(self,color=(0,1,0,0)):
        self.edge_col=(color[0]/255.0,color[1]/255.0,color[2]/255.0,color[3]/255.0)

    def set_back_color(self,color=(0.0,0.0,0.0,0.0)):
        self.back_col=(color[0]/255.0,color[1]/255.0,color[2]/255.0,color[3]/255.0)
        
    def initgl(self):
        try:
            self.initialized
        except:
            self.init_once()
        
    def set_view(self,orientation="iso"):
        self.Vscale=self.VscaleDefault
        self.mPanX=0.0
        self.mPanY=0.0
        self.scale_factor_last=1.0
        # Initialize the accumulated rotation matrix
        self.mAccumulatedRotation = numpy.identity(4,dtype=numpy.float32)
        if orientation == "front":
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -90, 1.0, 0.0, 0.0)
        elif orientation == "left":
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -90, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation,  90, 0.0, 0.0, 1.0)
        elif orientation == "right":
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -90, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -90, 0.0, 0.0, 1.0)
        elif orientation == "top":
            pass
        elif orientation == "bottom":
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, 180, 1.0, 0.0, 0.0)
        elif orientation == "back":
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation,  90, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, 180, 0.0, 1.0, 0.0)
        elif orientation == "iso0":    
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -90, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation,  45, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation,  45, 0.0, 0.0, -1.0)
        elif orientation == "iso1":    
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -90, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation,  45, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -45, 0.0, 0.0, -1.0)
        elif orientation == "iso2":    
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -90, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -45, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -45, 0.0, 0.0, -1.0)
        elif orientation == "iso3":    
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -90, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -45, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation,  45, 0.0, 0.0, -1.0)
        elif orientation == "iso4":
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation,  90, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, 180, 0.0, 1.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -45, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -45, 0.0, 0.0, -1.0)
        elif orientation == "iso5":
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation,  90, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, 180, 0.0, 1.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation,  45, 1.0, 0.0, 0.0)
            self.mAccumulatedRotation=self.rotate_matrix(self.mAccumulatedRotation, -45, 0.0, 0.0, -1.0)

    def init_VBO(self):
        self.ecoords=[]
        self.STLmesh  = mesh.Mesh(numpy.zeros(3, dtype=mesh.Mesh.dtype), remove_empty_areas=False)
        self.update_VBO()

    def Create_and_Compile_Shader(self):
        vertex_shader   = self.CompileShader(GL.GL_VERTEX_SHADER,  self.vertex_shader_text);
        fragment_shader = self.CompileShader(GL.GL_FRAGMENT_SHADER,self.fragment_shader_text);
        self.shader = GL.shaders.compileProgram(vertex_shader, fragment_shader)


    def Bounding_Sphere(self):
        val = 99999
        xmin = self.STLmesh.x.min()
        xmax = self.STLmesh.x.max()
        ymin = self.STLmesh.y.min()
        ymax = self.STLmesh.y.max()
        zmin = self.STLmesh.z.min()
        zmax = self.STLmesh.z.max()
        if xmax==0 and ymax==0 and zmax ==0:
            xmin =  val 
            xmax = -val
            ymin =  val
            ymax = -val
            zmin =  val
            zmax = -val
        
        for line in self.ecoords:
            xmin=min(xmin,line[0])
            xmax=max(xmax,line[0])
            ymin=min(ymin,line[1])
            ymax=max(ymax,line[1])
            zmin=min(zmin,line[2])
            zmax=max(zmax,line[2])

        if xmin == val:
            xmin=0
            xmax=0
        if ymin == val:
            ymin=0
            ymax=0
        if zmin == val:
            zmin=0
            zmax=0
            
        self.BOUNDS=(xmin,xmax,ymin,ymax,zmin,zmax)

        #returns x,y,z of center and radius
        Xcent = (xmin + xmax)/2.0
        Ycent = (ymin + ymax)/2.0
        Zcent = (zmin + zmax)/2.0
    
        DX = xmax - Xcent
        DY = ymax - Ycent
        DZ = zmax - Zcent
        
        RS = DX*DX + DY*DY + DZ*DZ                        
        Radius = math.sqrt(RS)
        bounds=[Xcent, Ycent, Zcent, Radius]
        return  bounds
        

    def update_VBO(self):
        #print("Update_VBO")
        #BSphere is x,y,z of center and radius
        self.BSphere = self.Bounding_Sphere()
        self.BSphere[3]=max(self.BSphere[3],math.sqrt(1+1+1))
        self.offset = numpy.array([self.BSphere[0:3]],dtype=numpy.float32)
        ################################################################################
        self.Vscale        = 1.0
        self.VscaleDefault = 1.0
        
        self.mLightPosInModelSpace = numpy.array([0.0, 0.0, 0.0, 1],dtype=numpy.float32)
        # Initialize the accumulated rotation matrix
        self.set_view("iso0")
        
        self.mPanX = 0.0
        self.mPanY = 0.0
        # Position the eye in front of the origin.
        eyeX = 0.0
        eyeY = 0.0
        eyeZ = 0.0 + self.BSphere[3]*5.0
        # We are looking toward the distance
        lookX = 0.0
        lookY = 0.0
        lookZ = 0.0
        # Set our up vector. This is where our head would be pointing were we holding the camera.
        upX = 0.0
        upY = 1.0
        upZ = 0.0
                
        # Set the view matrix. This matrix can be said to represent the camera position.
        # NOTE: In OpenGL 1, a ModelView matrix is used, which is a combination of a model and
        # view matrix. In OpenGL 2, we can keep track of these matrices separately if we choose.
        self.mViewMatrix = self.setLookAtM(eyeX, eyeY, eyeZ, lookX, lookY, lookZ, upX, upY, upZ)
        ################################################################################

        rows,cols = self.STLmesh.points.shape
        ########################################################################
        ###                       Create Triangles                           ###
        ########################################################################
        self.STL_vertices = numpy.reshape(self.STLmesh.points,(rows*3,3))-self.offset
        self.STL_normals  = numpy.zeros((rows*3,3),dtype=numpy.float32)
        #Normalize Normals
        for row in range(rows):
            L=math.sqrt(  self.STLmesh.normals[row][0]**2
                        + self.STLmesh.normals[row][1]**2
                        + self.STLmesh.normals[row][2]**2)
            if L==0:
                L=1
            self.STL_normals[row*3:row*3+3] = self.STLmesh.normals[row]/L
            
        self.STL_vertex_buffer  = VBO(self.STL_vertices, 'GL_STATIC_DRAW')
        self.STL_normal_buffer  = VBO(self.STL_normals,  'GL_STATIC_DRAW')

        ########################################################################
        ###                           Create Edges                           ###
        ########################################################################
        self.EDGE_vertices = numpy.zeros((rows*6,3),dtype=numpy.float32)
        self.EDGE_normals  = numpy.zeros((rows*6,3),dtype=numpy.float32)
        for row in range(rows):
            self.EDGE_vertices[row*6+0] = self.STL_vertices[row*3+0]
            self.EDGE_vertices[row*6+1] = self.STL_vertices[row*3+1]
            self.EDGE_vertices[row*6+2] = self.STL_vertices[row*3+1]
            self.EDGE_vertices[row*6+3] = self.STL_vertices[row*3+2]
            self.EDGE_vertices[row*6+4] = self.STL_vertices[row*3+2]
            self.EDGE_vertices[row*6+5] = self.STL_vertices[row*3+0]
            
            self.EDGE_normals[row*6+0]  = self.STL_normals[row*3+0]
            self.EDGE_normals[row*6+1]  = self.STL_normals[row*3+1]
            self.EDGE_normals[row*6+2]  = self.STL_normals[row*3+1]
            self.EDGE_normals[row*6+3]  = self.STL_normals[row*3+2]
            self.EDGE_normals[row*6+4]  = self.STL_normals[row*3+2]
            self.EDGE_normals[row*6+5]  = self.STL_normals[row*3+0]

        self.EDGE_vertex_buffer = VBO(self.EDGE_vertices,'GL_STATIC_DRAW')
        self.EDGE_normal_buffer = VBO(self.EDGE_normals, 'GL_STATIC_DRAW')


        ########################################################################
        ###                           Create PATHS                           ###
        ########################################################################
        self.PATH_list=[]
        for i in range(1,len(self.ecoords)):
            if self.ecoords[i][3]==self.ecoords[i-1][3]:
                self.PATH_list.append(self.ecoords[i-1][0:3])
                self.PATH_list.append(self.ecoords[i][0:3])

        if self.PATH_list !=  []:
            self.PATH_array = numpy.array(self.PATH_list,dtype=numpy.float32)-self.offset
            self.PATH_vertex_buffer = VBO(self.PATH_array, 'GL_STATIC_DRAW')
        else:
            self.PATH_vertex_buffer = None
            self.PATH_array = None



        ########################################################################
        ###                           Create Axes                            ###
        ########################################################################
        L_Axis=self.BSphere[3]
        self.AXIS_vertices = numpy.zeros((6,3),dtype=numpy.float32)
        self.AXIS_colors   = numpy.ones((6,4),dtype=numpy.float32)
        
        self.AXIS_vertices[0][:] = -self.offset
        self.AXIS_vertices[1][:] = numpy.array([L_Axis,0,0],dtype=numpy.float32)-self.offset
        self.AXIS_vertices[2][:] = -self.offset
        self.AXIS_vertices[3][:] = numpy.array([0,L_Axis,0],dtype=numpy.float32)-self.offset
        self.AXIS_vertices[4][:] = -self.offset
        self.AXIS_vertices[5][:] = numpy.array([0,0,L_Axis],dtype=numpy.float32)-self.offset
        
        self.AXIS_colors[0:2][:] = [1,0,0,1]
        self.AXIS_colors[2:4][:] = [0,1,0,1]
        self.AXIS_colors[4:6][:] = [0,0,1,1]

        self.AXIS_vertex_buffer = VBO(self.AXIS_vertices, 'GL_STATIC_DRAW')
        self.AXIS_color_buffer  = VBO(self.AXIS_colors,   'GL_STATIC_DRAW')
        
        ########################################################################
        ###                           Create Box                             ###
        ########################################################################
        xmin,xmax,ymin,ymax,zmin,zmax = self.BOUNDS
        #Xcent = (self.STLmesh.x.min() + self.STLmesh.x.max())/2.0
        #Ycent = (self.STLmesh.y.min() + self.STLmesh.y.max())/2.0
        #Zcent = (self.STLmesh.z.min() + self.STLmesh.z.max())/2.0
        Xcent = (xmin + xmax)/2.0
        Ycent = (ymin + ymax)/2.0
        Zcent = (zmin + zmax)/2.0
        
        self.BOX_vertices = numpy.zeros((24,3),dtype=numpy.float32)
        x = xmax - Xcent
        y = ymax - Ycent
        z = zmax - Zcent
        
        self.BOX_vertices[ 0][:] = [-x,-y,-z]
        self.BOX_vertices[ 1][:] = [ x,-y,-z]
        
        self.BOX_vertices[ 2][:] = [ x,-y,-z]
        self.BOX_vertices[ 3][:] = [ x, y,-z]

        self.BOX_vertices[ 4][:] = [ x, y,-z]
        self.BOX_vertices[ 5][:] = [-x, y,-z]

        self.BOX_vertices[ 6][:] = [-x, y,-z]
        self.BOX_vertices[ 7][:] = [-x,-y,-z]
        ###
        self.BOX_vertices[ 8][:] = [-x,-y, z]
        self.BOX_vertices[ 9][:] = [ x,-y, z]
        
        self.BOX_vertices[10][:] = [ x,-y, z]
        self.BOX_vertices[11][:] = [ x, y, z]

        self.BOX_vertices[12][:] = [ x, y, z]
        self.BOX_vertices[13][:] = [-x, y, z]

        self.BOX_vertices[14][:] = [-x, y, z]
        self.BOX_vertices[15][:] = [-x,-y, z]
        ##
        self.BOX_vertices[16][:] = [-x,-y,-z]
        self.BOX_vertices[17][:] = [-x,-y, z]
        
        self.BOX_vertices[18][:] = [ x,-y,-z]
        self.BOX_vertices[19][:] = [ x,-y, z]

        self.BOX_vertices[20][:] = [ x, y,-z]
        self.BOX_vertices[21][:] = [ x, y, z]

        self.BOX_vertices[22][:] = [-x, y,-z]
        self.BOX_vertices[23][:] = [-x, y, z]

        self.BOX_vertex_buffer = VBO(self.BOX_vertices, 'GL_STATIC_DRAW')
        
    def resize(self,w,h):
        self.configure( width = w, height = h )
        
               
    def redraw(self, mode=None):
        try:
            GL.glUseProgram(self.shader)
        except:
            print("Redraw Failed")
            self.init_once()
            return
        
        GL.glClear(GL.GL_COLOR_BUFFER_BIT | GL.GL_DEPTH_BUFFER_BIT);        

        # Set program handles for drawing.
        size_location         = GL.glGetUniformLocation(self.shader, 'u_Size')
        const_color_location  = GL.glGetUniformLocation(self.shader, 'u_const_color')
        const_normal_location = GL.glGetUniformLocation(self.shader, 'u_const_normal')
        color_location        = GL.glGetUniformLocation(self.shader, 'a_Color')
        mMVPMatrixHandle      = GL.glGetUniformLocation(self.shader, "u_MVPMatrix")
        mMVMatrixHandle       = GL.glGetUniformLocation(self.shader, "u_MVMatrix" )
        mLightPosHandle       = GL.glGetUniformLocation(self.shader, "u_LightPos" )

        ##########################################################################################################
        ###                                           Light Position                                           ###   
        ##########################################################################################################
        # Calculate position of the light. Push into the distance.
        mLightModelMatrix = numpy.identity(4,dtype=numpy.float32)
        mLightModelMatrix = self.translate_matrix(mLightModelMatrix,0,0, self.BSphere[3]*2.5 )
        mLightPosInWorldSpace = numpy.matmul(mLightModelMatrix, self.mLightPosInModelSpace)
        mLightPosInEyeSpace = numpy.matmul(self.mViewMatrix,mLightPosInWorldSpace)
        # Pass in the light position in eye space.
        GL.glUniform3f(mLightPosHandle, mLightPosInEyeSpace[0], mLightPosInEyeSpace[1], mLightPosInEyeSpace[2]);
        
        ##########################################################################################################
        ###                                           Rotate Model                                             ###   
        ##########################################################################################################
        # Translate the oject into the screen.
        mModelMatrix = numpy.identity(4,dtype=numpy.float32)
        
        # Set a matrix that contains the current rotation.
        mCurrentRotation = numpy.identity(4,dtype=numpy.float32)
        mCurrentRotation = self.rotate_matrix(mCurrentRotation, self.DH, 0.0, 1.0, 0.0)
        mCurrentRotation = self.rotate_matrix(mCurrentRotation, self.DV, 1.0, 0.0, 0.0)

        # Multiply the current rotation by the accumulated rotation, and then set the accumulated rotation to the result.
        self.mAccumulatedRotation = numpy.matmul(mCurrentRotation,self.mAccumulatedRotation)
        
        # Rotate the cube taking the overall rotation into account.         
        mModelMatrix = numpy.matmul(mModelMatrix, self.mAccumulatedRotation)

        self.DV=0.0
        self.DH=0.0
        ##########################################################################################################
        ratio  = float(self.width)/float(self.height)
        if ratio <=1.0:
            wmax = self.BSphere[3]
        else:
            wmax = self.BSphere[3]*ratio
        hmax = wmax/ratio

        ##  Zoom Stuff   ##
        multiplier       = 100.0
        scale_factor_min = .01
        scale_factor_max = 5.00
        vscale_min   = (scale_factor_min-1)*multiplier+1
        vscale_max   = (scale_factor_max-1)*multiplier+1
        self.Vscale  = max(vscale_min,self.Vscale)
        self.Vscale  = min(vscale_max,self.Vscale)
        scale_factor = (1 + (self.Vscale-1)/multiplier)
        #This adjusts the pan to keep the screen centered during zoom
        self.mPanX = self.mPanX*(self.scale_factor_last/scale_factor)
        self.mPanY = self.mPanY*(self.scale_factor_last/scale_factor)
        ## End Zoom Stuff ##

        left   = -wmax*(scale_factor) + self.mPanX * (wmax*(scale_factor))/float(self.width)*2.0
        right  =  wmax*(scale_factor) + self.mPanX * (wmax*(scale_factor))/float(self.width)*2.0
        bottom = -hmax*(scale_factor) + self.mPanY * (wmax*(scale_factor))/float(self.width)*2.0
        top    =  hmax*(scale_factor) + self.mPanY * (wmax*(scale_factor))/float(self.width)*2.0
        far    =  10.0 * min(wmax,hmax)
        near   = -far
        self.scale_factor_last = scale_factor
        
        ############################################################################
        
        if (right==left):
            right = right + 1.0
            print("right==left")
        if (top==bottom):
            top   = top   + 1.0
            print("top==bottom")
        if (near==far):
            far   = far   + 1.0
            print("near==far")

        mProjectionMatrix = self.orthoM(left, right, bottom, top, near, far)

        # This multiplies the view matrix by the model matrix, and stores
        # the result in the MVP matrix (which currently contains model * view).
        mMVMatrix = numpy.matmul(self.mViewMatrix,mModelMatrix)

        # Pass in the modelview matrix.
        GL.glUniformMatrix4fv(mMVMatrixHandle, 1, False, mMVMatrix.transpose(), 0)

        # This multiplies the modelview matrix by the projection matrix,
        # and stores the result in the MVP matrix (which now contains model * view * projection).
        mMVPMatrix = numpy.matmul(mProjectionMatrix,mMVMatrix)

        # Pass in the combined matrix.
        GL.glUniformMatrix4fv(mMVPMatrixHandle, 1, False, mMVPMatrix.transpose(), 0)

        GL.glUniform1f(size_location,self.BSphere[3])
        GL.glClearColor(self.back_col[0],self.back_col[1],self.back_col[2],self.back_col[3])
        
        #####################################################################
        ###                      Plot Triangles                           ###
        #####################################################################
        if self.draw_faces:
            try:
                # bind in our coordinates and indices into gpu ram
                self.STL_vertex_buffer.bind()
                GL.glVertexPointer(3, GL.GL_FLOAT, 0, None)
                self.STL_normal_buffer.bind()
                GL.glNormalPointer( GL.GL_FLOAT, 0, None)

                try:
                    GL.glEnableClientState(GL.GL_VERTEX_ARRAY)
                    GL.glEnableClientState(GL.GL_NORMAL_ARRAY)
                    
                    GL.glUniform1i(const_color_location,1)
                    GL.glUniform1i(const_normal_location,0)
                    GL.glUniform4f(color_location,self.tri_col[0],self.tri_col[1],self.tri_col[2],self.tri_col[3])
                    GL.glDrawArrays(GL.GL_TRIANGLES, 0, len(self.STL_vertices))
                finally:
                    self.STL_vertex_buffer.unbind()
                    self.STL_normal_buffer.unbind() 
                    GL.glDisableClientState(GL.GL_NORMAL_ARRAY)
                    GL.glDisableClientState(GL.GL_VERTEX_ARRAY)
            except:
                pass
            finally:
                pass

        #####################################################################
        ###                      Plot Edges                               ###
        #####################################################################
        
        try:
            self.EDGE_vertex_buffer.bind()
            GL.glVertexPointer(3, GL.GL_FLOAT, 0, None)
            self.EDGE_normal_buffer.bind()
            GL.glNormalPointer( GL.GL_FLOAT, 0, None)
            try:
                GL.glEnableClientState(GL.GL_VERTEX_ARRAY)
                GL.glEnableClientState(GL.GL_NORMAL_ARRAY)
                GL.glUniform1i(const_color_location,1)
                GL.glUniform1i(const_normal_location,0)
                if self.draw_edges:
                    GL.glLineWidth(1.5)
                    GL.glUniform4f(color_location,self.edge_col[0],self.edge_col[1],self.edge_col[2],self.edge_col[3])
                    GL.glDrawArrays(GL.GL_LINES, 0, len(self.EDGE_vertices))
                elif self.draw_faces:
                    pass
                    GL.glLineWidth(.001)
                    GL.glUniform4f(color_location,self.tri_col[0],self.tri_col[1],self.tri_col[2],self.tri_col[3])
                    GL.glDrawArrays(GL.GL_LINES, 0, len(self.EDGE_vertices))
            finally:
                self.EDGE_vertex_buffer.unbind()
                self.EDGE_normal_buffer.unbind() 
                GL.glDisableClientState(GL.GL_NORMAL_ARRAY)
                GL.glDisableClientState(GL.GL_VERTEX_ARRAY)
        except:
            pass
        finally:
            pass
        #####################################################################
        ###                       Plot PATHS                              ###
        #####################################################################
        if self.draw_paths:
            if self.PATH_list != []:       
                try:
                    self.PATH_vertex_buffer.bind()
                    GL.glVertexPointer(3, GL.GL_FLOAT, 0, None)
                    try:
                        GL.glEnableClientState(GL.GL_VERTEX_ARRAY)
                        GL.glLineWidth(.1)
                        GL.glUniform1i(const_color_location,1)
                        GL.glUniform1i(const_normal_location,1)
                        #GL.glUniform4f(color_location,0.5,1.0,0.5,1)
                        GL.glUniform4f(color_location,self.edge_col[0],self.edge_col[1],self.edge_col[2],self.edge_col[3])
                        GL.glDrawArrays(GL.GL_LINES, 0, len(self.PATH_list))
                    finally:
                        self.PATH_vertex_buffer.unbind()
                        GL.glDisableClientState(GL.GL_VERTEX_ARRAY)
                except:
                    pass
                finally:
                    pass
            
        #####################################################################
        ###                       Plot AXES                               ###
        #####################################################################
        if self.draw_axes:
            try:
                self.AXIS_vertex_buffer.bind()
                GL.glVertexPointer(3, GL.GL_FLOAT, 0, None)

                self.AXIS_color_buffer.bind()
                GL.glColorPointer( 4, GL.GL_FLOAT, 0, None)
                try:
                    GL.glEnableClientState(GL.GL_VERTEX_ARRAY)
                    GL.glEnableClientState(GL.GL_COLOR_ARRAY)
                    GL.glLineWidth(5)
                    GL.glUniform1i(const_color_location,0)
                    GL.glUniform1i(const_normal_location,1)
                    GL.glUniform4f(color_location,1,0,0,0)
                    GL.glDrawArrays(GL.GL_LINES, 0, len(self.AXIS_vertices))
                finally:
                    self.AXIS_vertex_buffer.unbind()
                    self.AXIS_color_buffer.unbind()
                    GL.glDisableClientState(GL.GL_VERTEX_ARRAY)
                    GL.glDisableClientState(GL.GL_COLOR_ARRAY)
            finally:
                pass

        #####################################################################
        ###                       Plot BOX                                ###
        #####################################################################
        if self.draw_box:
            try:
                self.BOX_vertex_buffer.bind()
                GL.glVertexPointer(3, GL.GL_FLOAT, 0, None)
                try:
                    GL.glEnableClientState(GL.GL_VERTEX_ARRAY)
                    GL.glLineWidth(.1)
                    GL.glUniform1i(const_color_location,1)
                    GL.glUniform1i(const_normal_location,1)
                    GL.glUniform4f(color_location,.5,.5,.5,1)
                    GL.glDrawArrays(GL.GL_LINES, 0, len(self.BOX_vertices))
                finally:
                    self.BOX_vertex_buffer.unbind()
                    GL.glDisableClientState(GL.GL_VERTEX_ARRAY)
            finally:
                pass
            
        #####################################################################

        GL.glUniform1i(const_normal_location,1)
        GL.glUniform1i(const_color_location,1)
        GL.glUniform4f(color_location,1,1,1,1)
        GL.glFlush()
        
        #####################################################################
        
    def CompileShader(self,type,source):
        shader=GL.glCreateShader(type)
        GL.glShaderSource(shader,source)
        GL.glCompileShader(shader)
        result=GL.glGetShaderiv(shader,GL.GL_COMPILE_STATUS)
        if (result!=1):
            raise Exception("CompileShader Failed:\n"+GL.glGetShaderInfoLog(shader))
        return shader

    # Translates matrix m by x, y, and z
    def translate_matrix(self,matrix,x,y,z):
        T=numpy.identity(4,dtype=numpy.float32)
        T[0][3]=x
        T[1][3]=y
        T[2][3]=z
        return numpy.matmul(matrix, T)


    # Rotates matrix by angle angle around the axis (x,y,z)
    def rotate_matrix(self, m, angle, x, y, z):
        sTemp = self.setRotateM(angle, x, y, z)
        return numpy.matmul(m, sTemp)


    # Creates a matrix for rotation by angle a around the axis (x, y, z)
    def setRotateM(self, a, x, y, z):
        rm = numpy.zeros((4,4),dtype=numpy.float32)
        rm[3][3]=1 #rm[15]= 1
        a *= (math.pi / 180.0)
        s = math.sin(a)
        c = math.cos(a)
        if (1.0 == x and 0.0 == y and 0.0 == z):
            rm[0][0] = 1 #0
            rm[1][1] = c #5
            rm[2][1] = s #6
            rm[1][2] =-s #9
            rm[2][2] = c #10
        elif (0.0 == x and 1.0 == y and 0.0 == z):
            rm[0][0] = c #0
            rm[2][0] =-s #2
            rm[1][1] = 1 #5
            rm[0][2] = s #8
            rm[2][2] = c #10
        elif (0.0 == x and 0.0 == y and 1.0 == z):
            rm[0][0] = c #0
            rm[1][0] = s #1
            rm[0][1] =-s #4
            rm[1][1] = c #5
            rm[2][2] = 1 #10
        else:
            len = self.length(x, y, z)
            if (1.0 != len):
                recipLen = 1.0 / len
                x *= recipLen
                y *= recipLen
                z *= recipLen
            nc = 1.0 - c
            xy = x * y
            yz = y * z
            zx = z * x
            xs = x * s
            ys = y * s
            zs = z * s
            rm[0][0] = x*x*nc +  c #0
            rm[1][0] =  xy*nc + zs #1
            rm[2][0] =  zx*nc - ys #2
            rm[0][1] =  xy*nc - zs #4
            rm[1][1] = y*y*nc +  c #5
            rm[2][1] =  yz*nc + xs #6
            rm[0][2] =  zx*nc + ys #8
            rm[1][2] =  yz*nc - xs #9
            rm[2][2] = z*z*nc +  c #10
        return rm

    def setLookAtM(self,     eyeX,    eyeY,    eyeZ,
                          centerX, centerY, centerZ,
                              upX,     upY,     upZ):
        fx = centerX - eyeX;
        fy = centerY - eyeY;
        fz = centerZ - eyeZ;
        # Normalize f
        rlf = 1.0 / self.length(fx, fy, fz)
        fx *= rlf
        fy *= rlf
        fz *= rlf
        # compute s = f x up (x means "cross product")
        sx = fy * upZ - fz * upY
        sy = fz * upX - fx * upZ
        sz = fx * upY - fy * upX
        # and normalize s
        rls = 1.0 / self.length(sx, sy, sz)
        sx *= rls
        sy *= rls
        sz *= rls
        # compute u = s x f
        ux = sy * fz - sz * fy
        uy = sz * fx - sx * fz
        uz = sx * fy - sy * fx
        
        rm = numpy.zeros((4,4),dtype=numpy.float32)
        rm[0][0] = sx
        rm[1][0] = ux
        rm[2][0] = -fx
        rm[3][0] = 0.0

        rm[0][1] = sy
        rm[1][1] = uy
        rm[2][1] = -fy
        rm[3][1] = 0.0

        rm[0][2] = sz
        rm[1][2] = uz
        rm[2][2] = -fz
        rm[3][2] = 0.0

        rm[0][3] = 0.0
        rm[1][3] = 0.0
        rm[2][3] = 0.0
        rm[3][3] = 1.0
        return self.translate_matrix(rm,-eyeX, -eyeY, -eyeZ)
        
    def length(self,x,y,z):
        return math.sqrt(x*x + y*y + z*z)

    # Computes an orthographic projection matrix.
    def orthoM(self, left, right, bottom, top, near, far):
        r_width  = 1.0 / (right - left)
        r_height = 1.0 / (top - bottom)
        r_depth  = 1.0 / (far - near)
        x =  2.0 * (r_width)
        y =  2.0 * (r_height)
        z = -2.0 * (r_depth)
        tx = -(right + left)   * r_width
        ty = -(top   + bottom) * r_height
        tz = -(far   + near  ) * r_depth
        rm = numpy.zeros((4,4),dtype=numpy.float32)

        rm[0][0] = x   #0
        rm[1][0] = 0.0 #1
        rm[2][0] = 0.0 #2
        rm[3][0] = 0.0 #3

        rm[0][1] = 0   #4
        rm[1][1] = y   #5
        rm[2][1] = 0.0 #6
        rm[3][1] = 0.0 #7

        rm[0][2] = 0.0 #8
        rm[1][2] = 0.0 #9
        rm[2][2] = z   #10
        rm[3][2] = 0.0 #11

        rm[0][3] = tx  #12
        rm[1][3] = ty  #13
        rm[2][3] = tz  #14
        rm[3][3] = 1.0 #15
        return rm


if __name__ == '__main__':
    pass
##################################
# 0  4  8  12
# 1  5  9  13
# 2  6  10 14
# 3  7  11 15
##            rm[0][0] = #0
##            rm[1][0] = #1
##            rm[2][0] = #2
##            rm[3][0] = #3
##
##            rm[0][1] = #4
##            rm[1][1] = #5
##            rm[2][1] = #6
##            rm[3][1] = #7
##
##            rm[0][2] = #8
##            rm[1][2] = #9
##            rm[2][2] = #10
##            rm[3][2] = #11
##
##            rm[0][3] = #12
##            rm[1][3] = #13
##            rm[2][3] = #14
##            rm[3][3] = #15
