#!/usr/bin/python
"""
    Scorch View (STL, DXF and G-Code viewer)
    
    Copyright (C) <2021-2026> <Scorch>              
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

    Version 0.01
    - Initial code
    
    Version 0.02
    - Updated to be compatable with Python 3.14 and later

"""
version = '0.02'
title_text = "Scorch View V"+version
QUIET = False
DEBUG = False

import sys
VERSION = sys.version_info[0]

if VERSION == 3:
    from tkinter import *
    from tkinter.filedialog import *
    import tkinter.messagebox
    MAXINT = sys.maxsize
    from tkinter import ttk
    from tkinter.colorchooser import askcolor
    def trace_variable(variable, callback):
        return variable.trace_add("write", callback)
    def trace_delete(variable,callback):
        return variable.trace_remove("write",callback) 
    
else:
    from Tkinter import *
    from tkFileDialog import *
    import tkMessageBox
    MAXINT = sys.maxint
    import ttk
    from tkColorChooser import askcolor
    def trace_variable(variable, callback):
        return variable.trace_variable("w", callback)
    def trace_delete(variable,callback):
        return variable.trace_vdelete("w",callback)

if VERSION < 3 and sys.version_info[1] < 6:
    def next(item):
        return item.next()
    
import os
import getopt
import webbrowser
import struct
import tkinterdnd2 #https://github.com/pmgagne/tkinterdnd2

from stl import mesh #pip install numpy-stl
from EZ_OpenGL_Frame import EZ_OpenGL_Frame

# I was getting a logger error
# Imorting logger and using the command below fixed that
# but expoosed a PyOpenGL_accelerate error
# removing PyOpenGL_accelerate fixed that error
#import logging
#logging.basicConfig()
#pip uninstall PyOpenGL_accelerate

# Hack to make it work with py2exe
try:
    from OpenGL.platform import win32
except AttributeError:
    pass

from g_code_library import G_Code_Rip
from dxf import DXF_CLASS
import math
import traceback


################################################################################
class Application(Frame):
    def __init__(self, master):
        Frame.__init__(self, master)
        self.w = 0
        self.h = 0
        frame = Frame(master)
        self.master = master
        self.x = -1
        self.y = -1            
        self.createWidgets()
        self.initComplete = 1
        self.View_Refresh()

    def createWidgets(self):
        self.initComplete = 0
        self.master.bind("<Configure>", self.Master_Configure) #used for resizing
        self.master.bind('<F1>'       , self.KEY_F1)
        self.master.bind('<F2>'       , self.KEY_F2)
        self.master.bind('<F5>'       , self.KEY_F5)

        self.show_faces   = BooleanVar()
        self.show_edges   = BooleanVar()
        self.show_box     = BooleanVar()
        self.show_axes    = BooleanVar()
        self.show_col     = BooleanVar()
        self.current_input_file = StringVar()
        self.statusMessage = StringVar()
        
        self.HOME_DIR = os.path.expanduser("~")
        self.statusMessage.set("Welcome to Scorch View")

        # PAN and ZOOM STUFF
        self.panx = 0
        self.pany = 0
        self.lastx = 0
        self.lasty = 0

        ###########################################################################
        #                         INITILIZE VARIABLES                             #
        #    if you want to change a default setting this is the place to do it   #
        ###########################################################################
        self.Apply_Default_Settings()
   
        ##########################################################################
        ###                     END INITILIZING VARIABLES                      ###
        ##########################################################################
        self.OpenGL_frame = EZ_OpenGL_Frame(self.master, width=580, height=460)
        self.OpenGL_frame.animate=1 #dont forget this, or your simulation would not animated

        self.OpenGL_frame.bind("<1>"     ,    self.LeftMouseClick)
        self.OpenGL_frame.bind("<B1-Motion>", self.LeftMouseMove)

        self.OpenGL_frame.bind("<2>"     ,    self.MiddleMouseClick)
        self.OpenGL_frame.bind("<B2-Motion>", self.MiddleMouseMove)
        
        self.OpenGL_frame.bind("<3>"     ,    self.RightMouseClick)
        self.OpenGL_frame.bind("<B3-Motion>", self.RightMouseMove)

        self.OpenGL_frame.bind('<MouseWheel>',    self.__wheel)  # zoom for Windows and MacOS, but not Linux
        self.OpenGL_frame.bind('<Button-5>',      self.__wheel)  # zoom for Linux, wheel scroll down
        self.OpenGL_frame.bind('<Button-4>',      self.__wheel)  # zoom for Linux, wheel scroll up

        ##########################################################################
        # make a Status Bar
        self.statusbar = Label(self.master, textvariable=self.statusMessage, bd=1, relief=SUNKEN , height=1)
        self.statusbar.pack(anchor=SW, fill=X, side=BOTTOM)
        

        # Buttons        
        self.Compare_Button     = Button(self.master,text="Reload File" ,command=self.reload_design_file)
        self.Open_New_Button    = Button(self.master,text="Open File"   ,command=self.Open_Model_File)
        self.Clear_Image_Button = Button(self.master,text="Clear Model" ,command=self.Clear_Model_memory)

        self.Front_Button  = Button(self.master,text="Front"  ,command=self.View_Front)
        self.Left_Button   = Button(self.master,text="Left"   ,command=self.View_Left)
        self.Right_Button  = Button(self.master,text="Right"  ,command=self.View_Right)
        self.Top_Button    = Button(self.master,text="Top"    ,command=self.View_Top)
        self.Bottom_Button = Button(self.master,text="Bott"   ,command=self.View_Bottom)
        self.Back_Button   = Button(self.master,text="Back"   ,command=self.View_Back)
        self.Iso0_Button   = Button(self.master,text="TFR"    ,command=self.View_Iso0, bg="grey")
        self.Iso1_Button   = Button(self.master,text="TLF"    ,command=self.View_Iso1, bg="grey")
        self.Iso2_Button   = Button(self.master,text="BLF"    ,command=self.View_Iso2, bg="grey")
        self.Iso3_Button   = Button(self.master,text="BFR"    ,command=self.View_Iso3, bg="grey")
        self.Iso4_Button   = Button(self.master,text="TRB"    ,command=self.View_Iso4, bg="grey")
        self.Iso5_Button   = Button(self.master,text="BRB"    ,command=self.View_Iso5, bg="grey")
        
        # Left Column #
        self.Label_Options = Label(self.master,text="Options:", anchor=W)


        self.Label_show_faces = Label(self.master,text="Show Faces")
        self.Checkbutton_show_faces = Checkbutton(self.master,text=" ", anchor=W)
        self.Checkbutton_show_faces.configure(variable=self.show_faces)
        trace_variable(self.show_faces, self.View_Refresh)

        self.Label_show_edges = Label(self.master,text="Show Edges")
        self.Checkbutton_show_edges = Checkbutton(self.master,text=" ", anchor=W)
        self.Checkbutton_show_edges.configure(variable=self.show_edges)
        trace_variable(self.show_edges, self.View_Refresh)

        self.Label_show_box = Label(self.master,text="Show Box")
        self.Checkbutton_show_box = Checkbutton(self.master,text=" ", anchor=W)
        self.Checkbutton_show_box.configure(variable=self.show_box)
        trace_variable(self.show_box, self.View_Refresh)

        self.Label_show_axes = Label(self.master,text="Show Axes")
        self.Checkbutton_show_axes = Checkbutton(self.master,text=" ", anchor=W)
        self.Checkbutton_show_axes.configure(variable=self.show_axes)
        trace_variable(self.show_axes, self.View_Refresh)
        
        self.separator1 = Frame(self.master, height=2, bd=1, relief=SUNKEN)
        self.separator2 = Frame(self.master, height=2, bd=1, relief=SUNKEN)
        self.separator3 = Frame(self.master, height=2, bd=1, relief=SUNKEN)
        self.separator4 = Frame(self.master, height=2, bd=1, relief=SUNKEN)        

        # End Left Column #


        # Make Menu Bar
        self.menuBar = Menu(self.master, relief = "raised", bd=2)

        top_File = Menu(self.menuBar, tearoff=0)
        top_File.add("command", label = "Open Settings File",  command = self.menu_Read_Settings)
        top_File.add("command", label = "Open Design File", command = self.Open_Model_File)

        top_File.add("command", label = "Exit", command = self.menu_File_Quit)
        self.menuBar.add("cascade", label="File", menu=top_File)


        top_View = Menu(self.menuBar, tearoff=0)
        top_View.add("command", label = "Refresh", command = self.View_Refresh)

        top_View.add_separator()
        top_View.add_checkbutton(label = "Show Faces", variable=self.show_faces,  command= self.View_Refresh)
        top_View.add_checkbutton(label = "Show Edges", variable=self.show_edges,  command= self.View_Refresh)
        top_View.add_checkbutton(label = "Show Box", variable=self.show_box,  command= self.View_Refresh)
        top_View.add_checkbutton(label = "Show Axes", variable=self.show_axes,  command= self.View_Refresh)
        top_View.add_checkbutton(label = "Show Left Column", variable=self.show_col,  command= self.View_Refresh)

        self.menuBar.add("cascade", label="View", menu=top_View)
        


        top_Settings = Menu(self.menuBar, tearoff=0)
        top_Settings.add("command", label = "General Settings", command = self.GEN_Settings_Window)

        self.menuBar.add("cascade", label="Settings", menu=top_Settings)

        top_Help = Menu(self.menuBar, tearoff=0)
        top_Help.add("command", label = "About", command = self.menu_Help_About)
        top_Help.add("command", label = "Help (Web Page)", command = self.menu_Help_Web)
        self.menuBar.add("cascade", label="Help", menu=top_Help)

        self.master.config(menu=self.menuBar)

        ##########################################################################
        #                  Config File and command line options                  #
        ##########################################################################
        config_file = "ScorchView.txt"
        home_config1 = os.path.expanduser("~") + "/" + config_file
        if ( os.path.isfile(config_file) ):
            self.Read_Settings(config_file)
        elif ( os.path.isfile(home_config1) ):
            self.Read_Settings(home_config1)

        opts, args = None, None
        try:
            opts, args = getopt.getopt(sys.argv[1:], "hdi:o",["help","debug", "input_file", "other_option"])
        except:
            fmessage('Unable interpret command line options')
            sys.exit()
        #print(args)
        #print(opts)
        if args!=[]:
            self.MODEL_FILE=args[0]
            self.current_input_file.set(args[0])
        for option, value in opts:
            if option in ('-h','--help'):
                fmessage(' ')
                fmessage('Usage: python ScorchView.py [-g file]')
                fmessage('-o    : unknown other option (also --other_option)')
                fmessage('-h    : print this help (also --help)\n')
                sys.exit()
            if option in ('-d','--debug'):
                global DEBUG
                DEBUG = True
            if option in ('-i','--input_file'):
                self.MODEL_FILE=value
                self.current_input_file.set(value)
            if option in ('-o','--other_option'):
                    pass
        ##########################################################################
        self.OpenGL_frame.drop_target_register(tkinterdnd2.DND_FILES)
        self.OpenGL_frame.dnd_bind('<<Drop>>', self.drop_new)
        ##############################################################

   ################################################################################
    def Apply_Default_Settings(self, dummy=None):
        ##########################################################################
        ###                     Establish Settings VARIABLES                   ###
        ##########################################################################
        self.settings={} #set up settings dictionary
        #settings[VARIABLE] = [TYPE,DEFAULT VALUE]
        self.settings['MODEL_COLOR']       =['TUPLE',        (255,255,000,255)]
        self.settings['EDGE_COLOR']        =['TUPLE',        (200,200,000,255)]
        self.settings['BACK_COLOR']        =['TUPLE',        (000,000,000,255)]
        self.settings['show_faces']        =['BOOLEANVAR',   1]
        self.settings['show_edges']        =['BOOLEANVAR',   1]
        self.settings['show_box']          =['BOOLEANVAR',   1]
        self.settings['show_axes']         =['BOOLEANVAR',   1]
        self.settings['show_col']          =['BOOLEANVAR',   1]
        self.settings['current_input_file']=['STRINGVAR',    "None"]
        self.settings['MODEL_FILE']        =['STRING',       os.path.expanduser("~")+"/None"]
        ##############################################################
        #                Apply Default Settings                      #
        ##############################################################
        for VARIABLE in self.settings:
            if (self.settings[VARIABLE][0]=='STRINGVAR') or \
               (self.settings[VARIABLE][0]=='BOOLEANVAR'):
                getattr(self,VARIABLE).set(self.settings[VARIABLE][1])
            else:
                setattr(self,VARIABLE, self.settings[VARIABLE][1])
        
                
    ##########################################################################
    def Write_Settings(self):
        global Zero
        settings_text = []
        settings_text.append('( ScorchView Settings: '+version+' )')
        settings_text.append('( by Scorch - 2021-2026 )')
        settings_text.append("(=========================================================)")
        for VARIABLE in self.settings:
            if (self.settings[VARIABLE][0]=='STRINGVAR'):
                settings_text.append('(ScorchView_set ' + VARIABLE + '  "%s" )'  %( getattr(self,VARIABLE).get() ))
            elif (self.settings[VARIABLE][0]=='BOOLEANVAR'):
                settings_text.append('(ScorchView_set ' + VARIABLE + '  %s )' %( int(getattr(self,VARIABLE).get()) ))
            elif (self.settings[VARIABLE][0]=='TUPLE'):
                value = ''
                for val in getattr(self,VARIABLE):
                    value = value + '%d '%(val)
                settings_text.append('(ScorchView_set ' + VARIABLE + '  %s )' %( value))
            else:
                #print(getattr(self,VARIABLE))
                settings_text.append('(ScorchView_set ' + VARIABLE + '  "%s" )' %( getattr(self,VARIABLE) ))
        settings_text.append("(=========================================================)")
        self.settings_text = settings_text
        return
    ##########################################################################

    def Read_Settings(self,filename):
        try:
            fin = open(filename,'r')
        except:
            fmessage("Unable to open settings file: %s" %(filename))
            return
        text_codes=[]
        ident = "ScorchView_set"
        for line in fin:
            if ident in line:
                try:
                    VARIABLE = line[line.find(ident):].split()[1]
                    TYPE     = self.settings[VARIABLE][0]
                except:
                    continue
                #print(VARIABLE,TYPE)
                if (TYPE=='STRING'):
                    setattr(self,VARIABLE, line[line.find(VARIABLE):].split("\042")[1])
                    #print(getattr(self,VARIABLE))
                    
                elif (TYPE=='STRINGVAR'):
                    getattr(self,VARIABLE).set(line[line.find(VARIABLE):].split("\042")[1])
                    #print(getattr(self,VARIABLE)).get()

                elif (TYPE=='BOOLEANVAR'):
                    getattr(self,VARIABLE).set(int(line[line.find(VARIABLE):].split()[1]))
                    #print(getattr(self,VARIABLE)).get()

                elif (TYPE=='TUPLE'):
                    color=[]
                    for num in line[line.find(VARIABLE):].split()[1:-1]:
                        color.append(int(num))
                    setattr(self,VARIABLE,tuple(color))
                    #print(getattr(self,VARIABLE))
        fin.close()
        temp_name, fileExtension = os.path.splitext(filename)
        file_base=os.path.basename(temp_name)
        if self.initComplete == 1:
            self.menu_Mode_Change()
    ##########################################################################


    def Quit_Click(self, event):
        self.statusMessage.set("Exiting!")
        root.destroy()

    def Close_Current_Window_Click(self):
        win_id=self.grab_current()
        win_id.destroy()
        
    def status_message(self,message_text,color='white'):
        self.statusbar.configure( bg = color )
        self.statusMessage.set(message_text)
        self.update_idletasks()

    def Write_Config_File(self, event):
        self.Write_Settings()
        config_file = "ScorchView.txt"
        configname_full = self.HOME_DIR + "/" + config_file

        win_id=self.grab_current()
        if ( os.path.isfile(configname_full) ):
            if not message_ask_ok_cancel("Replace", "Replace Exiting Configuration File?\n"+configname_full):
                try:
                    win_id.withdraw()
                    win_id.deiconify()
                except:
                    pass
                return
        try:
            fout = open(configname_full,'w')
        except:
            self.statusMessage.set("Unable to open file for writing: %s" %(configname_full))
            self.statusbar.configure( bg = 'red' )
            return
        for line in self.settings_text:
            try:
                fout.write(line+'\n')
            except:
                fout.write('(skipping line)\n')
        fout.close
        self.statusMessage.set("Configuration File Saved: %s" %(configname_full))
        self.statusbar.configure( bg = 'white' )


    def Clear_Model_memory(self):
        self.OpenGL_frame.init_VBO()
        self.statusMessage.set("Design Data Cleared")
        self.statusbar.configure( bg = 'white' )
        self.View_Refresh()

    def View_Front(self):
        self.OpenGL_frame.set_view("front")
    def View_Left(self):
        self.OpenGL_frame.set_view("left")
    def View_Right(self):
        self.OpenGL_frame.set_view("right")
    def View_Top(self):
        self.OpenGL_frame.set_view("top")
    def View_Bottom(self):
        self.OpenGL_frame.set_view("bottom")
    def View_Back(self):
        self.OpenGL_frame.set_view("back")
    def View_Iso0(self):
        self.OpenGL_frame.set_view("iso0")
    def View_Iso1(self):
        self.OpenGL_frame.set_view("iso1")
    def View_Iso2(self):
        self.OpenGL_frame.set_view("iso2")
    def View_Iso3(self):
        self.OpenGL_frame.set_view("iso3")
    def View_Iso4(self):
        self.OpenGL_frame.set_view("iso4")
    def View_Iso5(self):
        self.OpenGL_frame.set_view("iso5")

    def reload_design_file(self):
        self.Open_Model_File(fileselect=self.MODEL_FILE)
        
    def load_STL(self, input_file):
        self.statusbar.configure( bg = 'yellow' )
        self.statusMessage.set("Loading File: %s" %(input_file))
        self.master.update()
        try:
            your_mesh = mesh.Mesh.from_file(input_file)
            self.OpenGL_frame.STLmesh = your_mesh
            self.OpenGL_frame.update_VBO()
            self.statusMessage.set("STL Load Complete")
            self.statusbar.configure( bg = 'white' )
        except:
            self.statusMessage.set("STL Load Failed: %s" %(input_file))
            self.statusbar.configure( bg = 'red' )
            
        self.View_Refresh()

    #  Drag and Drop: Drop method
    def drop_new(self,event):
        try:
            root.focus_force()
        except:
            pass
        
        if event.data:
            files = self.OpenGL_frame.tk.splitlist(event.data)
            if VERSION == 3:
                filename = files[0]
            else:
                filename = files[0].decode("utf-8")
            #self.load_STL(input_file=filename)
            self.Open_Model_File(fileselect=filename)
                       
        return event.action
       
    def LeftMouseClick(self,event):
        self.pan_lastx = event.x
        self.pan_lasty = event.y
        
    def LeftMouseMove(self,event):
        x = event.x
        y = event.y
        self.OpenGL_frame.DH = self.OpenGL_frame.DH+(x-self.pan_lastx)
        self.OpenGL_frame.DV = self.OpenGL_frame.DV+(y-self.pan_lasty)
        self.pan_lastx = x
        self.pan_lasty = y

    def RightMouseClick(self,event):
        self.Rpan_lastx = event.x
        self.Rpan_lasty = event.y
        
    def RightMouseMove(self,event):
        x = event.x
        y = event.y
        self.OpenGL_frame.mPanX = self.OpenGL_frame.mPanX-float(x-self.Rpan_lastx)
        self.OpenGL_frame.mPanY = self.OpenGL_frame.mPanY+float(y-self.Rpan_lasty)
        self.Rpan_lastx = x
        self.Rpan_lasty = y

    def MiddleMouseClick(self,event):
        self.Mpan_lastx = event.x
        self.Mpan_lasty = event.y
        
    def MiddleMouseMove(self,event):
        y = event.y
        self.OpenGL_frame.Vscale = self.OpenGL_frame.Vscale+(y-self.Mpan_lasty)
        self.Mpan_lasty = y


    def __wheel(self, event):
        # mouse wheel
        #print(event.x,event.y,event.num,event.delta)
        # Respond to Linux (event.num) or Windows (event.delta) wheel event
        if event.num == 5 or event.delta == -120:  # scroll down, smaller
            self.OpenGL_frame.Vscale = self.OpenGL_frame.Vscale + 10
        if event.num == 4 or event.delta == 120:  # scroll up, bigger            
            self.OpenGL_frame.Vscale = self.OpenGL_frame.Vscale - 10
    ##########################################################################
    ##########################################################################

    def menu_Read_Settings(self):
        #if ( not os.path.isdir(init_dir) ):
        init_dir = os.path.expanduser("~")
        fileselect = askopenfilename(filetypes=[("Settings Files","*.txt"),\
                                                ("All Files","*")],\
                                                 initialdir=init_dir)
        if fileselect != '' and fileselect != ():
            self.Read_Settings(fileselect)


    def Open_Model_File(self,fileselect=None):
        if fileselect==None:
            init_dir = os.path.dirname(self.MODEL_FILE)
            if ( not os.path.isdir(init_dir) ):
                init_dir = os.path.expanduser("~")
            fileselect = askopenfilename(filetypes=[("Design Files", ("*.stl","*.dxf","*.ngc","*.tap")),
                                                    ("STL Files", ("*.stl")),\
                                                    ("DXF Files", ("*.dxf")),\
                                                    ("G-Code Files", ("*.ngc","*.tap")),\
                                                    ("All Files","*")],\
                                                    initialdir=init_dir)
        if fileselect != '' and fileselect != ():
            self.OpenGL_frame.init_VBO()
            self.MODEL_FILE = fileselect
            fileName, fileExtension = os.path.splitext(fileselect)
            TYPE=fileExtension.upper()
            if TYPE=='.STL':
                self.load_STL(input_file=fileselect)
            elif TYPE=='.DXF':
                self.Open_DXF(filemname=fileselect)
            else:
                self.Open_G_Code(filename=fileselect)

 
    ##########################################################################
    def Open_DXF(self,filemname):
        self.DXF_FILE = filemname
        dxf_import=DXF_CLASS()
        tolerance = .0005
        try:
            fd = open(self.DXF_FILE)
            dxf_import.GET_DXF_DATA(fd,lin_tol = tolerance,get_units=True,units=None)
            fd.seek(0)
            
            dxf_units = dxf_import.units
            if dxf_units=="Unitless":
                d = UnitsDialog(root)
                dxf_units = d.result
            if dxf_units=="Inches":
                dxf_scale = 1.0
            elif dxf_units=="Feet":
                dxf_scale = 12.0
            elif dxf_units=="Miles":
                dxf_scale = 5280.0*12.0
            elif dxf_units=="Millimeters":
                dxf_scale = 1.0/25.4
            elif dxf_units=="Centimeters":
                dxf_scale = 1.0/2.54
            elif dxf_units=="Meters":
                dxf_scale = 1.0/254.0
            elif dxf_units=="Kilometers":
                dxf_scale = 1.0/254000.0
            elif dxf_units=="Microinches":
                dxf_scale = 1.0/1000000.0
            elif dxf_units=="Mils":
                dxf_scale = 1.0/1000.0
            else:
                return    

            lin_tol = tolerance / dxf_scale
            dxf_import.GET_DXF_DATA(fd,lin_tol=lin_tol,get_units=False,units=None)
            fd.close()
        except Exception as e:
            msg1 = "DXF Load Failed:"
            msg2 = "%s" %(e)
            self.statusMessage.set((msg1+msg2).split("\n")[0] )
            self.statusbar.configure( bg = 'red' )
            message_box(msg1, msg2)
            debug_message(traceback.format_exc())
        except:
            fmessage("Unable To open Drawing Exchange File (DXF) file.")
            debug_message(traceback.format_exc())
            return
        
        dxf_coords = dxf_import.DXF_COORDS_GET(new_origin=False)        
        if dxf_import.dxf_messages != "":
            msg_split=dxf_import.dxf_messages.split("\n")
            msg_split.sort()
            msg_split.append("")
            mcnt=1
            msg_out = ""
            for i in range(1,len(msg_split)):
                if msg_split[i-1]==msg_split[i]:
                    mcnt=mcnt+1
                else:
                    if msg_split[i-1]!="":
                        msg_line = "%s (%d places)\n" %(msg_split[i-1],mcnt)
                        msg_out = msg_out + msg_line
                    mcnt=1
            message_box("DXF Import:",msg_out)
                    
        ##########################
        ###   Create ECOORDS   ###
        ##########################
        ecoords = self.make_ecoords(dxf_coords)
        self.OpenGL_frame.ecoords = ecoords
        self.OpenGL_frame.update_VBO()
        self.statusMessage.set("DXF Load Complete")
        self.statusbar.configure( bg = 'white' )
        self.View_Refresh()
    ##########################################################################
    def Open_G_Code(self,filename):
        g_rip = G_Code_Rip()
        try:
            MSG = g_rip.Read_G_Code(filename, XYarc2line = True, arc_angle=2, units="in", Accuracy="")
            Error_Text = ""
            if MSG!=[]:
                fmessage(MSG)

        except Exception as e:
            msg1 = "G-Code Load Failed:  "
            msg2 = "Filename: %s" %(filename)
            msg3 = "%s" %(e)
            self.statusMessage.set((msg1+msg3).split("\n")[0] )
            self.statusbar.configure( bg = 'red' )
            message_box(msg1, "%s\n%s" %(msg2,msg3))
            debug_message(traceback.format_exc())

            
        ecoords= g_rip.generate_tool_paths(g_rip.g_code_data)
        self.OpenGL_frame.ecoords = ecoords
        self.OpenGL_frame.update_VBO()
        self.statusMessage.set("G-Code Load Complete")
        self.statusbar.configure( bg = 'white' )
        self.View_Refresh()
        
    ##########################################################################

    def make_ecoords(self,coords,scale=1):
        #self.reset()
        self.len  = 0
        self.move = 0
        
        xmax, ymax = -1e10, -1e10
        xmin, ymin =  1e10,  1e10
        ecoords=[]
        Acc=.001
        oldx = oldy = -99990.0
        first_stroke = True
        loop=0
        z1=0.0
        z2=0.0
        for line in coords:
            XY = line
            x1 = XY[0]*scale
            y1 = XY[1]*scale
            x2 = XY[2]*scale
            y2 = XY[3]*scale
            dxline= x2-x1
            dyline= y2-y1
            len_line=math.sqrt(dxline*dxline + dyline*dyline)
            
            dx = oldx - x1
            dy = oldy - y1
            dist   = math.sqrt(dx*dx + dy*dy)
            # check and see if we need to move to a new discontinuous start point
            if (dist > Acc) or first_stroke:
                loop = loop+1
                ecoords.append([x1,y1,z1,loop])
                if not first_stroke:
                    self.move = self.move + dist
                first_stroke = False
                
            self.len = self.len + len_line
            ecoords.append([x2,y2,z2,loop])
            oldx, oldy = x2, y2
            xmax=max(xmax,x1,x2)
            ymax=max(ymax,y1,y2)
            xmin=min(xmin,x1,x2)
            ymin=min(ymin,y1,y2)
        bounds = (xmin,xmax,ymin,ymax)
        return ecoords

    
    def menu_File_Quit(self):
        if message_ask_ok_cancel("Exit", "Exiting...."):
            self.Quit_Click(None)

    def View_Refresh(self, varName=None, index=None, mode=None):
        dummy_event = Event()
        dummy_event.widget=self.master
        self.Master_Configure(dummy_event,1)
        


    def menu_Mode_Change(self,varName=None, index=None, mode=None):
        dummy_event = Event()
        dummy_event.widget=self.master
        self.Master_Configure(dummy_event,1)


    def menu_Help_About(self):
        application="Scorch View"
        about = "%s Version %s\n\n" %(application,version)
        about = about + "By Scorch.\n"
        about = about + "\163\143\157\162\143\150\100\163\143\157\162"
        about = about + "\143\150\167\157\162\153\163\056\143\157\155\n"
        about = about + "https://www.scorchworks.com/\n\n"
        try:
            python_version = "%d.%d.%d" %(sys.version_info.major,sys.version_info.minor,sys.version_info.micro)
        except:
            python_version = ""
        about = about + "Python "+python_version+" (%d bit)" %(struct.calcsize("P") * 8)
        message_box("About %s" %(application),about)
        

    def menu_Help_Web(self):
        #fmessage("No webpage yet.")
        webbrowser.open_new(r"http://www.scorchworks.com/Scorch_View/scorchview.html")

    def KEY_F1(self, event):
        self.menu_Help_About()

    def KEY_F2(self, event):
        self.GEN_Settings_Window()
        
    def KEY_F5(self, event):
        self.View_Refresh()
        
    def Master_Configure(self, event, update=0):
        if event.widget != self.master or self.initComplete != 1:
            return
        x = int(self.master.winfo_x())
        y = int(self.master.winfo_y())
        w = int(self.master.winfo_width())
        h = int(self.master.winfo_height())
        if (self.x, self.y) == (-1,-1):
            self.x, self.y = x,y
        if abs(self.w-w)>10 or abs(self.h-h)>10 or update==1:
            ###################################################
            #  Form changed Size (resized) adjust as required #
            ###################################################
            self.w=w
            self.h=h

            #####################################
            #          Left Column              #
            #####################################
            if self.show_col.get():
                left_col  = 185 # Set to 20 for no column
                wide_item = 160
                w_label=90
                w_entry=60-35
                w_units=35

                x_label_L=10
                x_entry_L=x_label_L+w_label+10
                x_units_L=x_entry_L+w_entry+5

                Yloc=6
                self.Label_show_faces.place(x=x_label_L, y=Yloc, width=w_label+20, height=21)
                self.Checkbutton_show_faces.place(x=x_entry_L+20, y=Yloc, width=w_entry+20, height=23)

                Yloc=Yloc+24
                self.Label_show_edges.place(x=x_label_L, y=Yloc, width=w_label+20, height=21)
                self.Checkbutton_show_edges.place(x=x_entry_L+20, y=Yloc, width=w_entry+20, height=23)

                Yloc=Yloc+24
                self.Label_show_box.place(x=x_label_L, y=Yloc, width=w_label+20, height=21)
                self.Checkbutton_show_box.place(x=x_entry_L+20, y=Yloc, width=w_entry+20, height=23)
                
                Yloc=Yloc+24
                self.Label_show_axes.place(x=x_label_L, y=Yloc, width=w_label+20, height=21)
                self.Checkbutton_show_axes.place(x=x_entry_L+20, y=Yloc, width=w_entry+20, height=23)
                
                Yloc=Yloc+24+12
                self.separator2.place(x=x_label_L, y=Yloc,width=wide_item, height=2)
                Yloc=Yloc+12
                
                self.Compare_Button.place(x=12, y=Yloc, width=wide_item, height=30)
                Yloc=Yloc+24+12

                self.Open_New_Button.place(x=12, y=Yloc, width=wide_item, height=30)
                Yloc=Yloc+24+12
                
                self.Clear_Image_Button.place(x=12, y=Yloc, width=wide_item, height=30)
                Yloc=Yloc+24+12

                vw = 40
                vh = 40
                self.Iso1_Button.place(  x=12,     y=Yloc, width=vw, height=vh)
                self.Top_Button.place(   x=12+vw,  y=Yloc, width=vw, height=vh)
                self.Iso0_Button.place(  x=12+vw*2,y=Yloc, width=vw, height=vh)
                self.Iso4_Button.place(  x=12+vw*3,y=Yloc, width=vw, height=vh)
                Yloc=Yloc+vh
                self.Left_Button.place(  x=12,     y=Yloc, width=vw, height=vh)
                self.Front_Button.place( x=12+vw,  y=Yloc, width=vw, height=vh)
                self.Right_Button.place( x=12+vw*2,y=Yloc, width=vw, height=vh)
                self.Back_Button.place(  x=12+vw*3,y=Yloc, width=vw, height=vh)
                
                Yloc=Yloc+vh
                self.Iso2_Button.place(  x=12,     y=Yloc, width=vw, height=vh)
                self.Bottom_Button.place(x=12+vw,  y=Yloc, width=vw, height=vh)
                self.Iso3_Button.place(  x=12+vw*2,y=Yloc, width=vw, height=vh)
                self.Iso5_Button.place(  x=12+vw*3,y=Yloc, width=vw, height=vh)
                
                Yloc=Yloc+24+12
        
            else:
                left_col = 20
                self.Label_show_faces.place_forget()
                self.Label_show_edges.place_forget()
                self.Label_show_box.place_forget()
                self.Label_show_axes.place_forget()
                self.Checkbutton_show_faces.place_forget()
                self.Checkbutton_show_edges.place_forget()
                self.Checkbutton_show_box.place_forget()
                self.Checkbutton_show_axes.place_forget()
                self.separator2.place_forget()
                self.Compare_Button.place_forget()
                self.Open_New_Button.place_forget()
                self.Clear_Image_Button.place_forget()
                self.Front_Button.place_forget()
                self.Left_Button.place_forget()
                self.Right_Button.place_forget()
                self.Top_Button.place_forget()
                self.Bottom_Button.place_forget()
                self.Back_Button.place_forget()
                self.Iso0_Button.place_forget()
                self.Iso1_Button.place_forget()
                self.Iso2_Button.place_forget()
                self.Iso3_Button.place_forget()
                self.Iso4_Button.place_forget()
                self.Iso5_Button.place_forget()
                
            canWidth  = self.w-(left_col+20)
            canHeight = self.h-45
            self.OpenGL_frame.resize(canWidth,canHeight)    
            self.OpenGL_frame.place(x=left_col, y=10)

        #self.View_Refresh()
        
        self.OpenGL_frame.draw_faces = self.show_faces.get()
        self.OpenGL_frame.draw_edges = self.show_edges.get()
        self.OpenGL_frame.draw_paths = self.show_edges.get()
        self.OpenGL_frame.draw_box = self.show_box.get()
        self.OpenGL_frame.draw_axes = self.show_axes.get()
        self.OpenGL_frame.set_tri_col(self.MODEL_COLOR)
        self.OpenGL_frame.set_edge_col(self.EDGE_COLOR)
        self.OpenGL_frame.set_back_color(self.BACK_COLOR)
        try:
            if (self.MODEL_FILE.find("None")<0):
                app.master.title(title_text+"   "+ self.MODEL_FILE)
        except:
            pass


    ################################################################################
    #                         General Settings Window                              #
    ################################################################################
    def GEN_Settings_Window(self):
        gen_settings = Toplevel(width=560, height=360)
        gen_settings.grab_set() # Use grab_set to prevent user input in the main window during calculations
        gen_settings.resizable(0,0)
        gen_settings.title('Settings')
        gen_settings.iconname("Settings")

        D_Yloc  = 6
        D_dY = 24
        xd_label_L = 12

        w_label=110
        w_entry=60
        w_units=35
        xd_entry_L=xd_label_L+w_label+10
        xd_units_L=xd_entry_L+w_entry+5
        ############################
        
        def get_MODEL_COLOR():
            color = askcolor("#%02x%02x%02x"%(self.MODEL_COLOR[0:3])) 
            if color[0] == None:
                return
            self.MODEL_COLOR = (int(color[0][0]), int(color[0][1]),int(color[0][2]),int(255))
            update_button_colors()
                
        def get_EDGE_COLOR():
            color = askcolor("#%02x%02x%02x"%(self.EDGE_COLOR[0:3]))
            if color[0] == None:
                return
            self.EDGE_COLOR = (int(color[0][0]), int(color[0][1]),int(color[0][2]),int(255))
            update_button_colors()

        def get_BACK_COLOR():
            color = askcolor("#%02x%02x%02x"%(self.BACK_COLOR[0:3]))
            if color[0] == None:
                return
            self.BACK_COLOR = (int(color[0][0]), int(color[0][1]),int(color[0][2]),int(255))
            update_button_colors()
            
        def update_button_colors():
            self.MODEL_COLOR_Button.config(bg="#%02x%02x%02x"%(self.MODEL_COLOR[0:3]))
            self.EDGE_COLOR_Button.config(bg="#%02x%02x%02x"%(self.EDGE_COLOR[0:3]))
            self.BACK_COLOR_Button.config(bg="#%02x%02x%02x"%(self.BACK_COLOR[0:3]))
            self.View_Refresh()
            
        def reset_settings(event=None):
            self.Apply_Default_Settings()
            update_button_colors()
            
        D_Yloc=D_Yloc+D_dY+10
        self.Label_MODEL_COLOR = Label(gen_settings,text="Model Color")
        self.Label_MODEL_COLOR.place(x=xd_label_L, y=D_Yloc, width=w_label, height=21)
        self.MODEL_COLOR_Button = Button(gen_settings,text='', command=get_MODEL_COLOR)
        self.MODEL_COLOR_Button.place(x=xd_entry_L, y=D_Yloc, width=50, height=21)

        D_Yloc=D_Yloc+D_dY+10
        self.Label_EDGE_COLOR = Label(gen_settings,text="Edge Color")
        self.Label_EDGE_COLOR.place(x=xd_label_L, y=D_Yloc, width=w_label, height=21)
        self.EDGE_COLOR_Button = Button(gen_settings,text='', command=get_EDGE_COLOR)
        self.EDGE_COLOR_Button.place(x=xd_entry_L, y=D_Yloc, width=50, height=21)

        D_Yloc=D_Yloc+D_dY+10
        self.Label_BACK_COLOR = Label(gen_settings,text="Background Color")
        self.Label_BACK_COLOR.place(x=xd_label_L, y=D_Yloc, width=w_label, height=21)
        self.BACK_COLOR_Button = Button(gen_settings,text='', command=get_BACK_COLOR)
        self.BACK_COLOR_Button.place(x=xd_entry_L, y=D_Yloc, width=50, height=21)

        self.MODEL_COLOR_Button.config(bg="#%02x%02x%02x"%(self.MODEL_COLOR[0:3]))
        self.EDGE_COLOR_Button.config( bg="#%02x%02x%02x"%(self.EDGE_COLOR[0:3] ))
        self.BACK_COLOR_Button.config( bg="#%02x%02x%02x"%(self.BACK_COLOR[0:3] ))

        ############################

        D_Yloc=D_Yloc+D_dY+10
        self.Label_SaveConfig = Label(gen_settings,text="Configuration File")
        self.Label_SaveConfig.place(x=xd_label_L, y=D_Yloc, width=113, height=21)

        self.GEN_SaveConfig = Button(gen_settings,text="Save")
        self.GEN_SaveConfig.place(x=xd_entry_L, y=D_Yloc, width=w_entry, height=21, anchor="nw")
        self.GEN_SaveConfig.bind("<ButtonRelease-1>", self.Write_Config_File)


        D_Yloc=D_Yloc+D_dY+10
        self.Label_SaveConfig = Label(gen_settings,text="Configuration")
        self.Label_SaveConfig.place(x=xd_label_L, y=D_Yloc, width=113, height=21)

        self.GEN_SaveConfig = Button(gen_settings,text="Reset")
        self.GEN_SaveConfig.place(x=xd_entry_L, y=D_Yloc, width=w_entry, height=21, anchor="nw")
        self.GEN_SaveConfig.bind("<ButtonRelease-1>", reset_settings)

        
        ## Buttons ##
        gen_settings.update_idletasks()
        Ybut=int(gen_settings.winfo_height())-30
        Xbut=int(gen_settings.winfo_width()/2)

        self.GEN_Close = Button(gen_settings,text="Close",command=self.Close_Current_Window_Click)
        self.GEN_Close.place(x=Xbut, y=Ybut, width=130, height=30, anchor="center")



################################################################################
#                         Debug Message Box                                    #
################################################################################
def debug_message(message):
    global DEBUG
    title = "Debug Message"
    if DEBUG:
        if VERSION == 3:
            tkinter.messagebox.showinfo(title,message)
        else:
            tkMessageBox.showinfo(title,message)
            pass
        
################################################################################
#             Function for outputting messages to different locations          #
#            depending on what options are enabled                             #
################################################################################
def fmessage(text,newline=True):
    global QUIET
    if (not QUIET):
        if newline==True:
            try:
                sys.stdout.write(text)
                sys.stdout.write("\n")
            except:
                pass
        else:
            try:
                sys.stdout.write(text)
            except:
                pass
            
################################################################################
#                               Message Box                                    #
################################################################################
def message_box(title,message):
    if VERSION == 3:
        tkinter.messagebox.showinfo(title,message)
    else:
        tkMessageBox.showinfo(title,message)
        pass
    
################################################################################
#                          Message Box ask OK/Cancel                           #
################################################################################
def message_ask_ok_cancel(title, mess):
    if VERSION == 3:
        result=tkinter.messagebox.askokcancel(title, mess)
    else:
        result=tkMessageBox.askokcancel(title, mess)
    return result

################################################################################
#                         Choose Units Dialog                                  #
################################################################################
if VERSION < 3:
    import tkSimpleDialog
else:
    import tkinter.simpledialog as tkSimpleDialog

class UnitsDialog(tkSimpleDialog.Dialog):
    def body(self, master):
        self.resizable(0,0)
        self.title('Units')
        self.iconname("Units")

        #try:
        #    self.iconbitmap(bitmap="@emblem64")
        #except:
        #    pass
        
        self.uom = StringVar()
        self.uom.set("Millimeters")

        Label(master, text="Select DXF Import Units:").grid(row=0)
        Radio_Units_IN = Radiobutton(master,text="Inches",        value="Inches")
        Radio_Units_MM = Radiobutton(master,text="Millimeters",   value="Millimeters")
        Radio_Units_CM = Radiobutton(master,text="Centimeters",   value="Centimeters")
        
        Radio_Units_IN.grid(row=1, sticky=W)
        Radio_Units_MM.grid(row=2, sticky=W)
        Radio_Units_CM.grid(row=3, sticky=W)

        Radio_Units_IN.configure(variable=self.uom)
        Radio_Units_MM.configure(variable=self.uom)
        Radio_Units_CM.configure(variable=self.uom)

    def apply(self):
        self.result = self.uom.get()
        return
    
################################################################################
#                          Startup Application                                 #
################################################################################
root = tkinterdnd2.Tk()
app = Application(root)
app.master.title(title_text)
app.master.iconname("Scorch View")
app.master.minsize(780,540)

################################## Set Icon  ########################################
Icon_Set=False
if getattr(sys, 'frozen', False):
    try:
        root.iconbitmap(default=sys.argv[0])
        Icon_Set=True
    except:
        Icon_Set=False
        
if not Icon_Set:
    try:
        scorch_ico_B64=b'R0lGODlhEAAQAIYAAA\
        AAABAQEBYWFhcXFxsbGyUlJSYmJikpKSwsLC4uLi8vLzExMTMzMzc3Nzg4ODk5OTs7Oz4+PkJCQkRERE\
        VFRUtLS0xMTE5OTlNTU1dXV1xcXGBgYGVlZWhoaGtra3FxcXR0dHh4eICAgISEhI+Pj5mZmZ2dnaKioq\
        Ojo62tra6urrS0tLi4uLm5ub29vcLCwsbGxsjIyMzMzM/Pz9PT09XV1dbW1tjY2Nzc3OHh4eLi4uXl5e\
        fn5+jo6Ovr6+/v7/Hx8fLy8vT09PX19fn5+fv7+/z8/P7+/v///wAAAAAAAAAAAAAAAAAAAAAAAAAAAA\
        AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA\
        AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA\
        AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEKAEkALAAAAAAQABAAQAj/AJMIFBhBQYAACRIkWbgwAA\
        4kEFEECACAxBAkGH8ESEKgBZIiAIQECBAjAA8kNwIkScKgQhAkRggAIJACCZIaJxgk2clgAY4OAAoEAO\
        ABCIIDSZIwkIHEBw0YFAAA6IGDCBIkLAhMyICka9cAKZCIRTLEBIMkaA0MSNGjSBEVIgpESEK3LgMCI1\
        aAWCFDA4EDSQInwaDACBEAImLwCAFARw4HFJJcgGADyZEAL3YQcMGBBpIjHx4EeIGkRoMFJgakWADABx\
        IkPwIgcIGkdm0AMJDo1g3jQBIBRZAINyKAwxEkyHEUSMIcwYYbEgwYmQGgyI8SD5Jo327hgIIAAQ5cBs\
        CQpHySgAA7'
        icon_im =PhotoImage(data=scorch_ico_B64, format='gif')
        root.call('wm', 'iconphoto', root._w, '-default', icon_im)
    except:
        pass
#####################################################################################

debug_message("Debuging is turned on.")
root.mainloop()
